#include "stdafx.h"
#include "MainFrm.h"
#include "RFApp.h"
#include "RFDoc.h"
#include "RFView.h"
#include "DiskController.h"
#include "DSP.h"
#include "XMLFile.h"

#include <fstream>

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

void sysTimeToFileName(SYSTEMTIME *st, CString &fileName)
{
	fileName.Format("%.4d-%.2d-%.2d %.2dh%.2dm%.2ds",
		st->wYear,
		st->wMonth,
		st->wDay,
		st->wHour,
		st->wMinute,
		st->wSecond );
}

bool DirectoryExists(CString &path) 
{
	DWORD dwAttrib = GetFileAttributes(path.GetBuffer());

	return (dwAttrib != INVALID_FILE_ATTRIBUTES && 
		   (dwAttrib & FILE_ATTRIBUTE_DIRECTORY));
}

// 2 ^ n
int pow2(int n) 
{
	int val = 0x1;

	while(n) {
		val <<= 1;
		n--;
	}

	return val;
}

// return log2(n)
int log2(int n) 
{
	int p = -1;
	while(n != 0u)
	{
		n >>= 1;
		++p;
	}
	return p;	
}

// CSHDoc
IMPLEMENT_DYNCREATE(CRFDoc, CDocument)

BEGIN_MESSAGE_MAP(CRFDoc, CDocument)
END_MESSAGE_MAP()

//
// CSHDoc construction/destruction
//
CRFDoc::CRFDoc()
{
	ippStaticInit();

	raw = ippsMalloc_32f(RAW_BUF_SIZE);
	rawshorts = ippsMalloc_16s(RAW_BUF_SIZE);
	overloaded = false;

	for(int i = 0; i < RAW_BUF_SIZE; i++) {
		raw[i] = 0.0f;
		rawshorts[i] = 0;
	}

	dsp.Resize(viewSettings.resolution, true);
	disk = new DiskController();
}

CRFDoc::~CRFDoc()
{
	ippsFree(raw);
	ippsFree(rawshorts);
	delete disk;
	device.Close();
}

//
// Open the bb60
//
BOOL CRFDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;
	
	device.Open();

	if(!device.IsOpen()) {
		MessageBoxA( 
			0,
			"No device connected.. Closing",
			_T("Fatal Error"),
			MB_OK | MB_ICONERROR 
			);		
		PostQuitMessage(-1);
	}

	programRunning = true;
	recording = false;
	reconfigure = true;

	mainThread = AfxBeginThread(
		ThreadEntry, 
		this,
		THREAD_PRIORITY_NORMAL,
		CREATE_SUSPENDED
		);
	mainThread->m_bAutoDelete = false;
	mainThread->ResumeThread();
	mainThread->GetMainWnd();

	return TRUE;
}

//
// Close the bb60 and main thread
//
void CRFDoc::OnCloseDocument()
{
	if(recording) {
		StartStopRecord();
	}

	DWORD result;
	programRunning = false;
	result = WaitForSingleObject(mainThread->m_hThread, 1000);
	
	if(result = WAIT_FAILED || result == WAIT_TIMEOUT) {
		TerminateThread(mainThread->m_hThread, -1);
	}

	delete mainThread;
	mainThread = NULL;

	// This has to be after the thread closing
	CDocument::OnCloseDocument();
}

//
// Not using serialization
//
void CRFDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// TODO: add storing code here
	}
	else
	{
		// TODO: add loading code here
	}
}

//
// Handy Dandy function to invalidate only our view
//
void CRFDoc::InvalidateView() const
{
	POSITION pos = this->GetFirstViewPosition();
	((CRFView*)this->GetNextView(pos))->Invalidate();
}

//
//
//
UINT CRFDoc::ThreadEntry(LPVOID param)
{
	return ((CRFDoc*)param)->MainThread();
}

//
// Main running thread
//
UINT CRFDoc::MainThread() 
{	
	int counter = 0;

	while(programRunning) {
		// Reconfigure check
		if(reconfigure) {
			dsp.Resize(viewSettings.resolution, true);
			device.Configure(settings);
			reconfigure = false;
		}

		// Get sweep
		device.GetSpectrum(raw, triggers);

		if(device.LastStatus() == bbADCOverflow) {
			overloaded = true;
		}

		// Update view
		if(++counter > viewSettings.refreshRate) {

			sigStats.Update(raw+16384, 4096);

			// Resize FFT buffers if changes resolution
			if(viewSettings.resolution != dsp.Order()) {
				dsp.Resize(viewSettings.resolution, true);
			}
			dsp.Process(raw /*+ 16384*/); // Do single FFT

			InvalidateView();
			counter = 0;
		}

		if(recording) {
			bbConvert_32f16s(raw, rawshorts, 15, settings.capture_len);
			disk->buffer(rawshorts);
			packetCount++;

			// Save trigger to xml file
			if(settings.captureTrigger) {
				// Add triggers to xml file
				int ti = 0;
				while(triggers[ti] && ti < 54) {
					// Appropriately adjust the triggers so the values run 
					//  continuously through the samples
					__int64 trigPos = 
						(__int64)((packetCount * 
						((true) ? 149504LL : 299008LL))
						+ triggers[ti]);
					CString trigPosStr;
					trigPosStr.Format("%lld", trigPos);
					xmlFile->AppendChild("Trigger");
					xmlFile->SetNodeValue(trigPosStr.GetBuffer());
					xmlFile->SelectParent();
					ti++;
				}
			}
		}
	}

	return -1;
}

//
// Update our live settings and trigger
//   a reconfigure
//
void CRFDoc::Reconfigure(const bbSettings_t &newSettings) 
{
	if(recording) {
		StartStopRecord();
	}

	settings = newSettings;
	reconfigure = true;
}

//
// Update view related settings
//
void CRFDoc::UpdateViewSettings(const viewSettings_t &vs)
{
	viewSettings = vs;
}

//
// Logic for both starting and stopping
//
bool CRFDoc::StartStopRecord()
{
	// Stop recording logic
	if(recording) {
		disk->stop();
		xmlFile->Print();
		delete xmlFile;
		recording = false;
		return true;
	}

	// Start recording logic
	// Create xml file here
	if(settings.path == initSaveDir) {
		AfxMessageBox("Invalid Directory Path");
		return false; // Still blank
	}
	// In the clear, set up file name
	CString fileName, fullPath;
	SYSTEMTIME sysTime;
	GetLocalTime(&sysTime);
	sysTimeToFileName(&sysTime, fileName);
	fullPath = settings.path + fileName;
	disk->setPath(fullPath.GetBuffer());
	xmlFile = new XMLFile(fullPath.GetBuffer());
	WriteOutXML();
	disk->beginStoring(settings.capture_len);
	AfxMessageBox(fullPath.GetBuffer());
	
	packetCount = 0;
	recording = true;

	return true;
}

void CRFDoc::WriteOutXML()
{
	CString str;

	settings.captureTitle.Remove(' ');
	xmlFile->AppendChild(settings.captureTitle.GetBuffer());
	xmlFile->AppendChild("Center");
	str.Format("%lf MHz", settings.center);
	xmlFile->SetNodeValue(str);
	xmlFile->SelectParent();
	xmlFile->AppendChild("Gain");
	str.Format("%d", settings.gain);
	xmlFile->SetNodeValue(str);
	xmlFile->SelectParent();
	xmlFile->AppendChild("Attenuation");
	str.Format("%d", settings.atten);
	xmlFile->SetNodeValue(str);
	xmlFile->SelectParent();
	xmlFile->AppendChild("Triggers");
}

//
// CSHDoc diagnostics
//
#ifdef _DEBUG
void CRFDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CRFDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG


// CRFDoc commands
